/*
* Copyright (C) 2008-2010, GlobFX Technologies, SARL.
* All Rights Reserved.
* This is UNPUBLISHED PROPRIETARY SOURCE CODE of GlobFX Technologies, SARL.
* The contents of this file may not be disclosed to third parties, copied or
* duplicated in any form, in whole or in part, without the prior written
* permission of GlobFX Technologies, SARL.
*/

package com.globfx.showroom
{
  import flash.display.Sprite;

  /**
   *  <p>The ShowRoom class represents the main ShowRoom API object.</p>
   *  <p>It should be created once at startup to establish the bridge between the Flash and PowerPoint.</p>
   */
  public class ShowRoom
  {
    private static var _showroom : ShowRoom = null;

    private static const _bridge_client_version : int = 1;
    
    private var _bridge_server_version : int = 0;
    private var _api : Object = null;

    private var _pres : Presentation = null;
    private var _slideshow : SlideShow = null;
    private var _shape : SwiffPlayerShape = null;
    private var _background : SwiffPlayerBackground = null;

    private var _showroom_version : String = null;
    private var _ppt_version : String = null;
    private var _ppt_release : int = 0;

    /////////////////////////////////////////////////////////////////////////
    
    /**
     *  <p>Creates a ShowRoom object and initializes the bridge between Flash and PowerPoint.</p>
     *  <p>If the Flash is not running in PowerPoint, returns <code>null</code>.</p>
     * 
     *  @return A ShowRoom object or <code>null</code> if the Flash is not running into PowerPoint.
     * 
     *  @throws Error If a ShowRoom object has already been created.
     *  
     */
    public static function create( root : flash.display.Sprite ) : ShowRoom
    {
      if( _showroom != null )
        throw new Error(); // already created
      
      _showroom= new ShowRoom();
      if( ! _showroom._init(root) )
      {
        _showroom._dispose();
        _showroom= null;
      }
      
      return _showroom;
    }
    
    /////////////////////////////////////////////////////////////////////////
    
    private function _init( root : flash.display.Sprite ) : Boolean
    {        
      try
      {
        _api= root.loaderInfo.loader.root["ShowRoom_CreateAPI"](_bridge_client_version);

        if( _api == null )
          return false;
        
        var info : * = _api.Init();
        if( info == null )
          return false;

        if( "error" in info && info.error != null && info.error.length > 0 )
        {
          trace("*** ShowRoom: error with init(): "+String(info.error) );
          return false;
        }
        
        _bridge_server_version= int(info.version);

        _showroom_version= String(info.showroom_version);
        _ppt_version= String(info.ppt_version);
        _ppt_release= int(info.ppt_release);
        
        _pres= new Presentation();
        if( !_pres._init(this,info) )
          return false;
        
        _slideshow= new SlideShow();
        if( !_slideshow._init(this,info) )
          return false;
        
        _shape= new SwiffPlayerShape();
        if( !_shape._init(this,info) )
          return false;

        _background= new SwiffPlayerBackground();
        if( !_background._init(this,info) )
          return false;
        
        return true;
      }
      catch( e : Error )
      {
      }

      return false;
    }
    
    /////////////////////////////////////////////////////////////////////////
    
    /**
     *  <p>Disposes a ShowRoom object.</p>
     * 
     */
    public function dispose() : void
    {
      _dispose();
      _showroom= null;
    }
    
    /////////////////////////////////////////////////////////////////////////
    
    private function _dispose() : void
    {
      if( _background != null )
      {
        _background._dispose();
        _background= null;
      }     
      
      if( _shape != null )
      {
        _shape._dispose();
        _shape= null;
      }     
      
      if( _slideshow != null )
      {
        _slideshow._dispose();
        _slideshow= null;
      }     
      
      if( _pres != null )
      {
        _pres._dispose();
        _pres= null;
      }     
      
      if( _api != null )
      {
        try
        {
          _api.Dispose();
        }
        catch( e : Error )
        {
        }
        _api= null;
      }

      _bridge_server_version= 0;
      _showroom_version= null;
      _ppt_version= null;
      _ppt_release= 0;
    }
    
    /////////////////////////////////////////////////////////////////////////
    
    /**
     * @private
     */
    internal function get api() : Object
    {
      return _api;
    }

    /////////////////////////////////////////////////////////////////////////
    
    /**
     *  The PowerPoint Presentation that is displayed by the Slide Show.
     */
    public function get presentation() : Presentation
    {
      return _pres;
    }
    
    /////////////////////////////////////////////////////////////////////////
    
    /**
     *  The PowerPoint Slide Show that is currently playing.
     */
    public function get slideShow() : SlideShow
    {
      return _slideshow;
    }
    
    /////////////////////////////////////////////////////////////////////////
    
    /**
     *  The PowerPoint Shape that represents this Flash in the slide.
     */
    public function get shape() : SwiffPlayerShape
    {
      return _shape;
    }
    
    /////////////////////////////////////////////////////////////////////////

    /**
     *  <p>The background bitmap behind the Swiff Player PowerPoint Shape during the slide show.</p>
     *  <p>This bitmap can be used only if the Flash is transparent in the slide. To make
     *     a Flash transparent, check the "Transparent" check box in the shape properties. If the
     *     shape is not transparent, this property returns <code>null</code>.</p>
     *
     */
    public function get background() : SwiffPlayerBackground
    {
      return _background;
    }
    
    /////////////////////////////////////////////////////////////////////////

    /**
     *  The version of ShowRoom that plays this Flash. ShowRoom version strings
     *  include major, minor, build and revision values, separated by dots,
     *  e.g. "2.0", or "2.1", or "2.0.1"
     */
    public function get version() : String
    {
      return _showroom_version;
    }
    
    /////////////////////////////////////////////////////////////////////////

    /**
     *  The version of PowerPoint that plays the slide show. Sample version
     *  string is "12.0.6500.5000" for PowerPoint 2007 SP2 for instance.
     */
    public function get pptVersion() : String
    {
      return _ppt_version;
    }

    /////////////////////////////////////////////////////////////////////////
    
    /**
     *  The release number of PowerPoint that plays the slide show.
     *  Release 11 is PowerPoint 2003.
     *  Release 12 is PowerPoint 2007.
     *  Release 14 is PowerPoint 2010.
     */
    public function get pptRelease() : int
    {
      return _ppt_release;
    }
    
    /////////////////////////////////////////////////////////////////////////
  }
}
