/*
* Copyright (C) 2008-2010, GlobFX Technologies, SARL.
* All Rights Reserved.
* This is UNPUBLISHED PROPRIETARY SOURCE CODE of GlobFX Technologies, SARL.
* The contents of this file may not be disclosed to third parties, copied or
* duplicated in any form, in whole or in part, without the prior written
* permission of GlobFX Technologies, SARL.
*/

package com.globfx.showroom
{
  import flash.events.Event;
  import flash.events.EventDispatcher;
  
  ///////////////////////////////////////////////////////////////////////////
  
  /**
   *  <p>Dispatched when the ShowRoom effect index has changed.</p>
   *  <p>The number of ShowRoom effects must have been specified in the
   *  PowerPoint ShowRoom UI ("Flash" Ribbon Tab or "Developer" Tab in the 
   *  properties dialog box).</p>
   *  <p>See <a href="http://www.globfx.com/support/showroom/flashdev/effects.php" target="_top">How to edit ShowRoom Effects</a>.</p>
   *
   *  @effectChanged com.globfx.showroom.EFFECT_CHANGED
   *
   */
  [Event(name="playStateChanged", type="flash.events.Event")]
  
  ///////////////////////////////////////////////////////////////////////////
  
  /**
   *  Dispatched when the Swiff Player Shape is removed from the slide during the slide show and disposed.
   *
   *  @effectChanged com.globfx.showroom.UNLOAD
   *
   */
  [Event(name="unload", type="flash.events.Event")]
  
  ///////////////////////////////////////////////////////////////////////////
  
  /**
   *  <p>The SwiffPlayerShape class represents the PowerPoint shape that holds Flash Player into the PowerPoint slide.</p>
   *  <p>It can be retrieved through the <code>ShowRoom.shape</code> property.</p>
   *  
   *  @see com.globfx.showroom.ShowRoom
   *  @see com.globfx.showroom.ShowRoom#shape
   */
  public class SwiffPlayerShape extends EventDispatcher
  {
    public static const EFFECT_CHANGED : String = "effectChanged";
    public static const UNLOAD : String = "unload";
    
    /////////////////////////////////////////////////////////////////////////
    
    private var _showroom : ShowRoom = null;
    
    private var _fullscreen : Boolean = false;
    
    private var _effect_index : int = -1;
    private var _effect_count : int = -1;
    
    /////////////////////////////////////////////////////////////////////////
    
    /**
     * @private
     */
    internal function _init( showroom : ShowRoom, info : * ) : Boolean
    {
      _showroom= showroom;
      
      _fullscreen= Boolean(info.fullscreen);
      _effect_index= int(info.effect_index);
      _effect_count= int(info.effect_count);
      
      _showroom.api.SetEffectIndex_Function= _onSetEffectIndex;
      _showroom.api.DisposeShape_Function= _onDisposeShape;
      
      return true;
    }
    
    /////////////////////////////////////////////////////////////////////////
    
    /**
     * @private
     */
    internal function _dispose() : void
    {
      if( _showroom != null )
      {
        _showroom.api.SetEffectIndex_Function= null;
        _showroom.api.DisposeShape_Function= null;
      
        _fullscreen= false;
        
        _effect_index= -1;
        _effect_count= -1;
        
        _showroom= null;
      }
    }
    
    /////////////////////////////////////////////////////////////////////////
    
    private function _onSetEffectIndex( effect_index : int ) : void
    {
      if( _effect_index == effect_index )
        return;
      
      _effect_index= effect_index;
      dispatchEvent( new Event(EFFECT_CHANGED) );
    }
    
    /////////////////////////////////////////////////////////////////////////
    
    private function _onDisposeShape() : void
    {
      dispatchEvent( new Event(UNLOAD) );
    }
    
    /////////////////////////////////////////////////////////////////////////
    
    /**
     *  <p>Returns the current ShowRoom effect to execute.</p>
     *
     *  <p>Returns -1 if the shape is not yet initialized, 0 for the first effect, etc.</p>
     *
     *  <p>The number of ShowRoom effects must have been specified in the
     *  PowerPoint ShowRoom UI ("Flash" Ribbon Tab or "Developer" Tab in the 
     *  properties dialog box).</p>
     *  <p>See <a href="http://www.globfx.com/support/showroom/flashdev/effects.php" target="_top">How to edit ShowRoom Effects</a>.</p>
     */
    public function get effectIndex() : int
    {
      return _effect_index;
    }
    
    /////////////////////////////////////////////////////////////////////////
    
    /**
     *  <p>Returns the number of ShowRoom effects applied to this shape.</p>
     *
     *  <p>This number is specified in PowerPoint in the shapes properties.</p>
     *
     *  <p>The number of ShowRoom effects must have been specified in the
     *  PowerPoint ShowRoom UI ("Flash" Ribbon Tab or "Developer" Tab in the 
     *  properties dialog box).</p>
     *  <p>See <a href="http://www.globfx.com/support/showroom/flashdev/effects.php" target="_top">How to edit ShowRoom Effects</a>.</p>
     */
    public function get numEffects() : int
    {
      return _effect_count;
    }
    
    /////////////////////////////////////////////////////////////////////////
    
    /**
     *  A flag that indicates whether the Flash is displayed in full screen mode or into the slide.
     */
    public function get fullScreen() : Boolean
    {
      return _fullscreen;
    }
    
    /////////////////////////////////////////////////////////////////////////
    
    /**
     *  <p>When displayed in full screen mode, closes the player to continue the slide show.</p>
     *
     */
    public function closeFullScreenPlayer() : void
    {
      if( _fullscreen )
        _showroom.api.CloseFullScreenPlayer();
    }
    
    /////////////////////////////////////////////////////////////////////////
  }
}
