/*
* Copyright (C) 2008-2010, GlobFX Technologies, SARL.
* All Rights Reserved.
* This is UNPUBLISHED PROPRIETARY SOURCE CODE of GlobFX Technologies, SARL.
* The contents of this file may not be disclosed to third parties, copied or
* duplicated in any form, in whole or in part, without the prior written
* permission of GlobFX Technologies, SARL.
*/

package com.globfx.showroom
{
  /**
   *  <p>The Presentation class represents the PowerPoint presentation that is currently displayed by the slide show.</p>
   *  <p>It can be retrieved through the <code>ShowRoom.presentation</code> property.</p>
   *  
   *  @see com.globfx.showroom.ShowRoom
   *  @see com.globfx.showroom.ShowRoom#presentation
   */
  public class Presentation
  {
    private var _showroom : ShowRoom = null;
    
    private var _name : String = null;
    
    private var _num_slides : int = 0;
    private var _slides : Array = null;
    
    /////////////////////////////////////////////////////////////////////////
    
    /**
     * @private
     */
    internal function _init( showroom : ShowRoom, info : * ) : Boolean
    {
      _showroom= showroom;
      
      _name= String(info.pres_name);
      _num_slides= int(info.slide_count);
      _slides= new Array();
      
      return true;
    }
    
    /////////////////////////////////////////////////////////////////////////
    
    /**
     * @private
     */
    internal function _dispose() : void
    {
      for( var i : int = 0; i < _slides.length; ++i )
      {
        var slide : Slide = _slides[i];
        if( slide != null )
        {
          slide._dispose();
          _slides[i]= null;
        }
      }
      
      _slides= null;
      _num_slides= 0;
      
      _showroom= null;
    }
    
    /////////////////////////////////////////////////////////////////////////
    
    /**
     * @private
     */
    internal function get showroom() : ShowRoom
    {
      return _showroom;
    }
    
    /////////////////////////////////////////////////////////////////////////
    
    /**
     *  <p>Returns the name of the presentation.</p>
     *  <p>The name a presentation could be "Presentation1" if it has not been saved yet,
     *     or or "MyPres.pptx" if it has been saved.</p>
     */
    public function get name() : String
    {
      return _name;
    }
    
    /////////////////////////////////////////////////////////////////////////
    
    /**
     *  <p>Returns the total number of slides of the presentation.</p>
     *  <p>This number does not necessarily equals the number of slides
     *     of the slide show.</p>
     *
     *  @see SlideShow#numSlides
     */
    public function get numSlides() : int
    {
      return _num_slides;
    }
    
    /////////////////////////////////////////////////////////////////////////
    
    /**
     *  <p>Returns the specified slide of the presentation.</p>
     *  <p>Slide numbers go from 1 to <code>numSlides</code> inclusive.</p>
     *
     *  @param number The number of the slide object.
     *
     *  @return The slide object for the specified slide number.
     *
     *  @see #numSlides
     */
    public function getSlide( number : int ) : Slide
    {
      if( number <= 0 || number > _num_slides )
        return null;
      
      var slide : Slide = _slides[number];
      if( slide == null )
      {
        slide= new Slide();
        slide._init( this, number );
        _slides[number]= slide;
      }
      
      return slide;
    }
    
    /////////////////////////////////////////////////////////////////////////
    
    /**
     *  <p>Returns the slide with the specified title in the presentation.</p>
     *  <p>If several slides have this title, the first one is returned.</p>
     *
     *  @param title The title of the slide object.
     *
     *  @return The slide object that has the specified title.
     */
    public function findSlideByTitle( title : String ) : Slide
    {
      if( title == null )
        return null;
      var number : int = int( _showroom.api.FindSlideByTitle(title) );
      if( number == 0 )
        return null;
      return getSlide(number);
    }
    
    /////////////////////////////////////////////////////////////////////////
  }
}
