/*
* Copyright (C) 2008-2010, GlobFX Technologies, SARL.
* All Rights Reserved.
* This is UNPUBLISHED PROPRIETARY SOURCE CODE of GlobFX Technologies, SARL.
* The contents of this file may not be disclosed to third parties, copied or
* duplicated in any form, in whole or in part, without the prior written
* permission of GlobFX Technologies, SARL.
*/

package com.globfx.showroom
{
  import flash.events.Event;
  import flash.events.EventDispatcher;

  ///////////////////////////////////////////////////////////////////////////

  /**
   *  Dispatched when the slide show is paused and resumed.
   *
   *  @eventType com.globfx.showroom.PLAY_STATE_CHANGED
   *
   */
  [Event(name="playStateChanged", type="flash.events.Event")]

  ///////////////////////////////////////////////////////////////////////////

  /**
   *  <p>The SlideShow class represents the PowerPoint slide show that is currently playing.</p>
   *  <p>It can be retrieved through the <code>ShowRoom.slideShow</code> property.</p>
   *  
   *  @see com.globfx.showroom.ShowRoom
   *  @see com.globfx.showroom.ShowRoom#slideShow
   */
  public class SlideShow extends EventDispatcher
  {
    private var _showroom : ShowRoom = null;
    
    private var _current_slide_number : int = 0;
    private var _start_slide_number : int = 0;
    private var _end_slide_number : int = 0;
    private var _paused : Boolean = false;

    private var _start_date : Date = null;
    private var _history : Array = null;
    
    public static const PLAY_STATE_CHANGED : String = "playStateChanged";
    
    /////////////////////////////////////////////////////////////////////////
    
    /**
    * @private
    */
    internal function _init( showroom : ShowRoom, info : * ) : Boolean
    {
      _showroom= showroom;
      
      _current_slide_number= int(info.slide_number);
      _start_slide_number= int(info.start_slide);
      _end_slide_number= int(info.end_slide);
      _start_date= new Date( Number(info.start_date) );
      _paused= Boolean(info.paused);

      _showroom.api.SetPlayState_Function= _onSetPlayState;
      
      return true;
    }
    
    /////////////////////////////////////////////////////////////////////////
    
    /**
    * @private
    */
    internal function _dispose() : void
    {
      if( _showroom != null )
        {
        _showroom.api.SetPlayState_Function= null;

        _history= null;
        _current_slide_number= 0;
        _start_slide_number= 0;
        _end_slide_number= 0;
        _paused= false;      
        
        _showroom= null;
        }
    }
    
    /////////////////////////////////////////////////////////////////////////

    private function _onSetPlayState( paused : Boolean ) : void
    {
      if( _paused == paused )
        return;
      
      _paused= paused;
      dispatchEvent( new Event(PLAY_STATE_CHANGED) );
    }
    
    /////////////////////////////////////////////////////////////////////////
    
	  /**
	   *  A flag that indicates whether the slide show is paused or playing.
	   */
    public function get paused() : Boolean
    {
      return _paused;
    }

    /////////////////////////////////////////////////////////////////////////
    
    /**
     *  Ends the slide show.
     */
    public function endSlideShow() : void
    {
      _showroom.api.EndSlideShow();
    }
    
    /////////////////////////////////////////////////////////////////////////
    
    /**
     *  Executes the next PowerPoint effect or jump to the next slide if there is no 
     *  more effect to run in the current slide.
     */
    public function next() : void
    {
      _showroom.api.Next();
    }
    
    /**
     *  Jump to the beginning of the previous effect or jump to the previous slide if there is no 
     *  more effect to unroll before.
     */
    public function previous() : void
    {
      _showroom.api.Previous();
    }
    
    /////////////////////////////////////////////////////////////////////////
    
    /**
     *  <p>Jump to the next slide of the slide show.</p>
     *  <p>Does nothing if the current slide is the starting slide of the slide show.</p>
     */
    public function gotoNextSlide() : void
    {
      if( _current_slide_number == _end_slide_number )
        return;
      gotoSlideNumber( _current_slide_number + 1 );
    }
    
    /**
     *  <p>Jump to the previous slide of the slide show.</p>
     *  <p>Does nothing if the current slide is the ending slide of the slide show.</p>
     */
    public function gotoPreviousSlide() : void
    {
      if( _current_slide_number == _start_slide_number )
        return;
      gotoSlideNumber( _current_slide_number - 1 );
    }
    
    /////////////////////////////////////////////////////////////////////////
    
    /**
     *  Jump to the first slide of the slide show.
     */
    public function gotoFirstSlide() : void
    {
      _showroom.api.GotoFirstSlide();
    }
    
    /**
     *  Jump to the last slide of the slide show.
     */
    public function gotoLastSlide() : void
    {
      _showroom.api.GotoLastSlide();
    }
    
    /////////////////////////////////////////////////////////////////////////
    
    /**
     *  <p>Jump to the specified slide number.</p>
     *  <p>If the specified slide cannot be reached (for example,
     *     not in the slide show), the function does nothing.</p>
     *  <p>For example, the first slide number of the slide show is <code>ShowRoom.slideShow.startSlide.number</code>.</p>
     *
     *  @param number The slide number to jump to.
     *
     *  @see #currentSlide
     *  @see #startSlide
     *  @see #endSlide
     */
    public function gotoSlideNumber( number : int ) : void
    {
      if( number < _start_slide_number || number > _end_slide_number )
        return;
      _showroom.api.GotoSlideNumber(number);
    }
    
    /////////////////////////////////////////////////////////////////////////

    /**
     *  Returns the current slide.
     */
    public function get currentSlide() : Slide
    {
      return _showroom.presentation.getSlide(_current_slide_number);
    }
    
    /////////////////////////////////////////////////////////////////////////
    
    /**
     *  Returns the starting slide of the slide show.
     */
    public function get startingSlide() : Slide
    {
      return _showroom.presentation.getSlide(_start_slide_number);
    }
    
    /////////////////////////////////////////////////////////////////////////
    
    /**
     *  Returns the ending slide of the slide show.
     */
    public function get endingSlide() : Slide
    {
      return _showroom.presentation.getSlide(_end_slide_number);
    }
    
    /////////////////////////////////////////////////////////////////////////
    
    /**
     *  <p>Returns the total number of slides of the slide show.</p>
     *  <p>This number does not necessarily equals the number of slides
     *     of the presentation. It can be a subset.</p>
     */
    public function get numSlides() : int
    {
      return _end_slide_number - _start_slide_number + 1;
    }
    
    /////////////////////////////////////////////////////////////////////////
    
    /**
     *  Returns the date and time the slide show was first started.
     */
    public function get startDate() : Date
    {
      return _start_date;
    }
    
    /////////////////////////////////////////////////////////////////////////
    
    /**
     *  <p>Returns the slide show history.</p>
     *  <p>The slide show history contains the list of all previously visited slides.
     *     Also recorded are the date and time the slide was entered as well as 
     *     the elapsed time on the slide.</p>
     *  <p>The first item of the array (index 0) represents the current slide,
     *     the next item (index 1) is the previously visited slide in the past, etc.</p>
     *
     *  @return An array of VisitedSlide objects.
     *
     *  @see VisitedSlide
     */
    public function get history() : Array
    {
      if( _history == null )
        {
        var res : * = _showroom.api.GetHistory();
        if( res == null )
          return null;

        var r_slide : Array = res.slide;
        var r_entered : Array = res.entered;
        var r_elapsed : Array = res.elapsed;
        
        var length : int = r_slide.length;
        var pres : Presentation = _showroom.presentation;

        _history= new Array();
                
        for( var i : int = 0; i < length; ++i )
          {
          var slide_number : int = int(r_slide[i]);
          var date_entered : Date = new Date( Number(r_entered[i]) );
          var elapsed : Number = Number(r_elapsed[i]) / 1000.0;
          
          var vs : VisitedSlide = VisitedSlide._create( pres.getSlide(slide_number),
                                                        date_entered,
                                                        elapsed );
          _history.push(vs);
          }
        }
        
      return _history;
    }
    
    /////////////////////////////////////////////////////////////////////////
  }
}
